library ieee;
use ieee.std_logic_1164.all;
package cpu_bus_pkg is
   generic(
      masters_max_jg : positive;
      slaves_max_jg  : positive;
      awidth_jg      : positive;
      dwidth_jg      : positive
   );
   
   subtype  masters_total_jrt  is natural range masters_max_jg - 1 downto 0;

   constant slaves_total_jc: positive := slaves_max_jg   -- individual slaves
                                       + masters_max_jg  -- master's slave return
                                       + 1;              -- arbiter's slave return
   subtype  slaves_total_jrt   is natural range slaves_total_jc - 1 downto 0;

   subtype  addr_jrt           is natural range awidth_jg - 1 downto 0;
   subtype  data_jrt           is natural range dwidth_jg - 1 downto 0;

   subtype  master_ctrl_vst    is std_logic_vector(masters_total_jrt);
   subtype  slave_ctrl_vst     is std_logic_vector(slaves_total_jrt);

   subtype  addr_vst           is std_logic_vector(addr_jrt);
   subtype  data_vst           is std_logic_vector(data_jrt);

   type master_r is                      -- Master record
      record
         addr_vl : addr_vst;             -- Address bus to slave devices
         data_vl : data_vst;             -- Data bus from master to slave
         we_l    : std_logic;            -- Write enable from master to slave
         en_vl   : slave_ctrl_vst;       -- Slave enables from master to slave
      end record master_r;

   type slave_r is                       -- Slave record
      record
         data_vl : data_vst;             -- Data bus from slave to master
         ack_l   : std_logic;            -- Acknowledge to master from slave
         err_l   : std_logic;            -- Error report to master from slave
      end record slave_r;

   interface slave_i is
      signal slave_rl : slave_r;
   end interface slave_i;

   interface slave_a is
      array(slaves_total_jrt) of slave_i;-- Array of slave interfaces

   interface arbiter_i is                -- Arbiter bundle
      signal master_rl   : master_r;     -- Master record to the arbiter
      signal bus_req_l   : std_logic;    -- Bus request from master
      signal bus_grnt_l  : std_logic;    -- Bus grants to master
   end interface arbiter_i;

   interface arbiter_a is 
      array(masters_total_jrt) of arbiter_i;-- Array of arbiter interfaces

   interface cpu_bus_i is                -- Bus interface record
      bundle arbiter_al : arbiter_a;     -- Arbiter array of arbiter master
                                         -- control records
      signal master_rl  : master_r;      -- Master record to slaves
      bundle slave_al   : slave_a;       -- Array of slave records to master
   end record cpu_bus_i;

   port view slave_ivw of cpu_bus_i is
      generic(
         slave_id_jg : slaves_total_jrt
      );
      port(
         signal master_rl : in     master_r;
         signal slave_rl  : buffer slave_r
      );
      port map(
         master_rl => master_rl,
         slave_rl  => slave_al(slave_id_jg)
      );
   end port view slave_ivw;

   port view arbiter_v_in of arbiter_i is
      mode map (master_rl => buffer, bus_req_l => buffer, bus_grnt_l => in);
--      This is equivalent to
--      port(
--         signal master_rl  : buffer master_r;
--         signal bus_req_l  : buffer std_logic;
--         signal bus_grnt_l : in     std_logic
--      );
   end port view arbiter_v_in;

   port view arbiter_v_out of arbiter_i is
      mode map (master_rl => in, bus_req_l => in, bus_grnt_l => buffer);
   end port view arbiter_v_out;

   port view master_ivw of cpu_bus_i is
      generic(
         slave_id_jg  : slaves_total_jrt;
         master_id_jg : masters_total_jrt
      );
      port(
         bundle arbiter_rl : view   arbiter_v_in;
         signal master_rl  : in     master_r;
         bundle slave_al   : view   slave_a
                              mode map (slave_id_jg => buffer, others => in)
      );
      port map(
         arbiter_rl => arbiter_al(master_id_jg),
         master_rl =>  master_rl,
         slave_al =>   slave_al
      };
   end port view master_ivw;

   port view a_arbiter_avw of arbiter_a is
      mode map(
         others => view    arbiter_v_out
      );
   end port view a_arbiter_avw;

   port view arbiter_ivw of cpu_bus_i is
      generic(
         slave_id_jg  : slaves_total_jrt
      );
      port(
         bundle arbiter_al  : view   a_arbiter_avw;
         signal master_rl   : buffer master_r;
         signal slave_rl    : buffer slave_r
      );
      port map(
         arbiter_al => arbiter_al,
         master_rl  => master_rl,
         slave_rl   => slave_al(slave_id_jg)
      );
   end port view arbiter_grnt_ivw;
end package cpu_bus_pkg;
